

// ---------- MAPI Types -----------

// all types found in MAPI TNEF attachment
const MAPI_UNSPECIFIED    = 0x0000; // MAPI Unspecified
const MAPI_NULL           = 0x0001; // MAPI null property
const MAPI_SHORT          = 0x0002; // MAPI short (signed 16 bits)
const MAPI_INT            = 0x0003; // MAPI integer (signed 32 bits)
const MAPI_FLOAT          = 0x0004; // MAPI float (4 bytes)
const MAPI_DOUBLE         = 0x0005; // MAPI double
const MAPI_CURRENCY       = 0x0006; // MAPI currency (64 bits)
const MAPI_APPTIME        = 0x0007; // MAPI application time
const MAPI_ERROR          = 0x000a; // MAPI error (32 bits)
const MAPI_BOOLEAN        = 0x000b; // MAPI boolean (16 bits)
const MAPI_OBJECT         = 0x000d; // MAPI embedded object
const MAPI_INT64	  = 0x0014; // MAPI 8 byte signed int
const MAPI_STRING         = 0x001e; // MAPI string
const MAPI_UNICODE_STRING = 0x001f; // MAPI unicode-string (null terminated)
const MAPI_SYSTIME        = 0x0040; // MAPI time (64 bits)
const MAPI_CLSID          = 0x0048; // MAPI OLE GUID
const MAPI_BINARY         = 0x0102; // MAPI binary
const MAPI_MULTI_STRING   = 0x101F; // MAPI multiple string


function mattr_type_to_string( mattr_type ) {
  switch( mattr_type ) {
  case MAPI_UNSPECIFIED: return "UNSPECIFIED";
  case MAPI_NULL: return "NULL";
  case MAPI_SHORT: return "SHORT";
  case MAPI_INT: return "INT";
  case MAPI_FLOAT: return "FLOAT";
  case MAPI_DOUBLE: return "DOUBLE";
  case MAPI_CURRENCY: return "CURRENCY";
  case MAPI_APPTIME: return "APPTIME";
  case MAPI_ERROR: return "ERROR";
  case MAPI_BOOLEAN: return "BOOLEAN";
  case MAPI_OBJECT: return "OBJECT";
  case MAPI_INT64: return "INT64";
  case MAPI_STRING: return "STRING";
  case MAPI_UNICODE_STRING: return "UNICODE_STRING";
  case MAPI_SYSTIME: return "SYSTIME";
  case MAPI_CLSID: return "CLSID";
  case MAPI_BINARY: return "BINARY";
  case MAPI_MULTI_STRING: return "MULTI_STRING";
  default: return( to_hex( mattr_type, 2 ) );
  }
}

// ---------- MAPI Names -----------

// names of some attributes found in TNEF file
const MAPI_ACKNOWLEDGEMENT_MODE		= 0x0001;
const MAPI_TEMPLATE_DATA		= 0x0001;
const MAPI_MEETING_ATTENDEE_CRITICAL_CHANGE = 0x0001;
const MAPI_ALTERNATE_RECEIPIANT_ALLOWED	= 0x0002;
const MAPI_MEETING_WHERE		= 0x0002;
const MAPI_AUTHORIZING_USERS		= 0x0003;
const MAPI_MEETING_GLOBAL_OBJECT_ID	= 0x0003;
const MAPI_AUTO_FORWARD_COMMENT		= 0x0004;
const MAPI_SCRIPT_DATA			= 0x0004;
const MAPI_MEETING_IS_SILENT		= 0x0004;
const MAPI_AUTO_FORWARDED		= 0x0005;
const MAPI_MEETING_IS_RECURRING		= 0x0005;
const MAPI_CONTENT_CONFIDENTIALITY_ALGORITHM_ID = 0x0006;
const MAPI_MEETING_REQUIRED_ATTENDEES	= 0x0006;
const MAPI_CONTENT_CORRELATOR		= 0x0007;
const MAPI_MEETING_OPTIONAL_ATTENDEES	= 0x0007;
const MAPI_CONTENT_IDENTIFIER		= 0x0008;
const MAPI_MEETING_RESOURCE_ATTENDEES	= 0x0008;
const MAPI_MEETING_DELEGATE_MAIL	= 0x0009;
const MAPI_CONTENT_RETURN_REQUESTED	= 0x000A;
const MAPI_MEETING_IS_EXCEPTION		= 0x000A;
const MAPI_CONVERSATION_KEY		= 0x000B;
const MAPI_MEETING_SINGLE_INVITE	= 0x000B;
const MAPI_MEETING_TIME_ZONE		= 0x000C;
const MAPI_CONVERSION_EITS		= 0x000C;
const MAPI_DEFERRED_DELIVERY_TIME	= 0x000F;
const MAPI_DELIVER_TIME			= 0x0010;
const MAPI_DISCARD_REASON		= 0x0011;
const MAPI_DISCLOSURE_OF_RECIPIENTS	= 0x0012;
const MAPI_DL_EXPANSION_HISTORY		= 0x0013;
const MAPI_MEETING_MONTH_INTERVAL	= 0x0013;
const MAPI_DL_EXPANSION_PROHIBITED	= 0x0014;
const MAPI_EXPIRY_TIME			= 0x0015;
const MAPI_IMPLICIT_CONVERSION_PROHIBITED = 0x0016;
const MAPI_IMPORTANCE			= 0x0017;
const MAPI_IPM_ID			= 0x0018;
const MAPI_LATEST_DELIVERY_TIME		= 0x0019;
const MAPI_MESSAGE_CLASS		= 0x001A;
const MAPI_MEETING_OWNER_CRITICAL_CHANGE = 0x001A;
const MAPI_MESSAGE_DELIVERY_ID		= 0x001B;
const MAPI_MEETING_CALENDAR_TYPE	= 0x001C;
const MAPI_MEETING_ALL_ATTENDEES	= 0x001D;
const MAPI_MESSAGE_SECURITY_LABEL	= 0x001E;
const MAPI_OBSOLETED_IPMS		= 0x001F;
const MAPI_ORIGINALLY_INTENDED_RECIPIENT_NAME = 0x0020;
const MAPI_ORIGINAL_EITS		= 0x0021;
const MAPI_ORIGINATOR_CERTIFICATE	= 0x0022;
const MAPI_ORIGINATOR_DELIVERY_REPORT_REQUESTED = 0x0023;
const MAPI_MEETING_CLEAN_GLOBAL_OBJECT_ID = 0x0023;
const MAPI_ORIGINATOR_RETURN_ADDRESS	= 0x0024;
const MAPI_MEETING_MESSAGE_CLASS	= 0x0024;
const MAPI_PARENT_KEY			= 0x0025;
const MAPI_PRIORITY			= 0x0026;
const MAPI_MEETING_TYPE			= 0x0026;
const MAPI_MEETING_OLD_LOCATION		= 0x0028;
const MAPI_READ_RECEIPT_REQUESTED	= 0x0029;
const MAPI_RECIPIENT_REASSIGNMENT_PROHIBITED = 0x002B;
const MAPI_RELATED_IPMS			= 0x002D;
const MAPI_ORIGINAL_SENSITIVITY		= 0x002E;
const MAPI_REPORT_TAG			= 0x0031;
const MAPI_SENSITIVITY			= 0x0036;
const MAPI_SUBJECT			= 0x0037;
const MAPI_CLIENT_SUBMIT_TIME		= 0x0039;
const MAPI_SUBJECT_PREFIX		= 0x003D;
const MAPI_RECEIVED_BY_NAME		= 0x0040;
const MAPI_SENT_REPRESENTING_ENTRYID	= 0x0041;
const MAPI_SENT_REPRESENTING_NAME	= 0x0042;
const MAPI_RCVD_REPRESENTING_ENTRYID	= 0x0043;
const MAPI_RCVD_REPRESENTING_NAME	= 0x0044;
const MAPI_REPORT_ENTRYID		= 0x0045;
const MAPI_READ_RECEIPT_ENTRYID		= 0x0046;
const MAPI_MESSAGE_SUBMISSION_ID	= 0x0047;
const MAPI_PROVIDER_SUBMIT_TIME		= 0x0048;
const MAPI_ORIGINAL_SUBJECT		= 0x0049;
const MAPI_DISC_VAL			= 0x004A;
const MAPI_ORIG_MESSAGE_CLASS		= 0x004B;
const MAPI_ORIGINAL_AUTHOR_ENTRYID	= 0x004C;
const MAPI_ORIGINAL_AUTHOR_NAME		= 0x004D;
const MAPI_MESSAGE_TO_ME		= 0x0057;
const MAPI_MESSAGE_CC_ME		= 0x0058;
const MAPI_ORIG_SENT_REPRESENTING_NAME	= 0x005D;
const MAPI_ORIG_SENT_REPRESENTING_ENTRYID = 0x005E;
const MAPI_START_DATE			= 0x0060;
const MAPI_END_DATE			= 0x0061;
const MAPI_OWNER_APPT_ID		= 0x0062;
const MAPI_RESPONSE_REQUESTED		= 0x0063;
const MAPI_SENT_REPRESENTING_ADDR_TYPE	= 0x0064;
const MAPI_SENT_REPRESENTING_EMAIL	= 0x0065;
const MAPI_ORIG_SENT_REPRESENTING_EMAIL	= 0x0069;
const MAPI_CONVERSATION_TOPIC		= 0x0070;
const MAPI_CONVERSATION_INDEX		= 0x0071;
const MAPI_RECEIVED_BY_ADDRTYPE		= 0x0075;
const MAPI_RECEIVED_BY_EMAIL_ADDRESS	= 0x0076;
const MAPI_TRANSPORT_MESSAGE_HEADERS	= 0x007D;
const MAPI_TNEF_CORRELATION_KEY		= 0x007F;
const MAPI_EXPLICIT_CONVERSION		= 0x0C01;
const MAPI_MESSAGE_TOKEN		= 0x0C03;
const MAPI_REPLY_REQUESTED		= 0x0C17;
const MAPI_SENDER_NAME			= 0x0C1A;
const MAPI_SENDER_SEARCH_KEY		= 0x0C1D;
const MAPI_SENDER_EMAIL_ADDRESS		= 0x0C1F;
const MAPI_DELETE_AFTER_SUBMIT		= 0x0E01;
const MAPI_DISPLAY_BCC			= 0x0E02;
const MAPI_DISPLAY_CC			= 0x0E03;
const MAPI_DISPLAY_TO			= 0x0E04;
const MAPI_PARENT_DISPLAY		= 0x0E05;
const MAPI_MESSAGE_DELIVERY_TIME	= 0x0E06;
const MAPI_MESSAGE_FLAGS		= 0x0E07;
const MAPI_MESSAGE_SIZE			= 0x0E08;
const MAPI_PARENT_ENTRYID		= 0x0E09;
const MAPI_SENTMAIL_ENTRYID		= 0x0E0A;
const MAPI_MESSAGE_RECIPIENTS		= 0x0E12;
const MAPI_MESSAGE_ATTACHMENTS		= 0x0E13;
const MAPI_SUBMIT_FLAGS			= 0x0E14;
const MAPI_HASATTACH			= 0x0E1B;
const MAPI_BODY_CRC			= 0x0E1C;
const MAPI_NORMALIZED_SUBJECT		= 0x0E1D;
const MAPI_RTF_IN_SYNC			= 0x0E1F;
const MAPI_ATTACH_SIZE			= 0x0E20;
const MAPI_ATTACH_NUM			= 0x0E21;
const MAPI_PREPROCESS			= 0x0E22;
const MAPI_PRIMARY_SEND_ACCOUNT		= 0x0E28;
const MAPI_NEXT_SEND_ACCOUNT		= 0x0E29;
const MAPI_ACCESS			= 0x0FF4;
const MAPI_ROW_TYPE			= 0x0FF5;
const MAPI_INSTANCE_KEY			= 0x0FF6;
const MAPI_ACCESS_LEVEL			= 0x0FF7;
const MAPI_MAPPING_SIGNATURE		= 0x0FF8;
const MAPI_RECORD_TYPE			= 0x0FF9;
const MAPI_MESSAGE_TYPE			= 0x0FFA;
const MAPI_OBJECT_TYPE			= 0x0FFE;
const MAPI_ENTRY_ID			= 0x0FFF;
const MAPI_BODY				= 0x1000;
const MAPI_REPORT_TEXT			= 0x1001;
const MAPI_RTF_SYNC_BODY_CRC		= 0x1006;
const MAPI_RTF_SYNC_BODY_COUNT		= 0x1007;
const MAPI_RTF_SYNC_BODY_TAG		= 0x1008;
const MAPI_RTF_COMPRESSED		= 0x1009;
const MAPI_RTF_SYNC_PREFIX_COUNT	= 0x1010;
const MAPI_RTF_SYNC_TRAILING_COUNT	= 0x1011;
const MAPI_BODY_HTML			= 0x1013;
const MAPI_SMTP_MESSAGE_ID		= 0x1035;
const MAPI_CONFLICT_ITEMS		= 0x1098;
const MAPI_URL_COMP_NAME		= 0x10F3;
const MAPI_ATTRIBUTE_READ_ONLY		= 0x10F6;
const MAPI_CATEGORIES			= 0x2328;
const MAPI_DISPLAY_NAME			= 0x3001;
const MAPI_EMAIL_ADDRESS		= 0x3003;
const MAPI_CREATION_TIME		= 0x3007;
const MAPI_MODIFICATION_TIME		= 0x3008;
const MAPI_SEARCH_KEY			= 0x300B;
const MAPI_STORE_SUPPORT_MASK		= 0x340D;
const MAPI_MDB_PROVIDER			= 0x3414;
const MAPI_CONTAINER_FLAGS		= 0x3600;
const MAPI_FOLDER_TYPE			= 0x3601;
const MAPI_CONTENT_COUNT		= 0x3602;
const MAPI_CONTENT_UNREAD		= 0x3603;
const MAPI_CONTAINER_HIERARCHY		= 0x360E;
const MAPI_CONTAINER_CONTENTS		= 0x360F;
const MAPI_FOLDER_ASSOCIATED_CONTENTS	= 0x3610;
const MAPI_CONTAINER_CLASS		= 0x3613;
const MAPI_CONTAINER_MODIFY_VERSION	= 0x3614;
const MAPI_ATTACHMENT_X400_PARAMETERS	= 0x3700;
const MAPI_ATTACH_DATA_OBJ		= 0x3701;
const MAPI_ATTACH_ENCODING		= 0x3702;
const MAPI_ATTACH_EXTENSION		= 0x3703;
const MAPI_ATTACH_FILENAME		= 0x3704;
const MAPI_ATTACH_METHOD		= 0x3705;
const MAPI_ATTACH_LONG_FILENAME		= 0x3707;
const MAPI_ATTACH_PATHNAME		= 0x3708;
const MAPI_ATTACH_RENDERING		= 0x3709;
const MAPI_ATTACH_TAG			= 0x370A;
const MAPI_ATTACHMENT_RENDERING_POSITION = 0x370B;
const MAPI_ATTACH_TRANSPORT_NAME	= 0x370C;
const MAPI_ATTACH_LONG_PATHNAME		= 0x370D;
const MAPI_ATTACH_MIME_TAG		= 0x370E;
const MAPI_ATTACH_ADDITIONAL_INFO	= 0x370F;
const MAPI_ATTACH_MIME_SEQUENCE		= 0x3710;
const MAPI_ATTACH_CONTENT_ID		= 0x3712;
const MAPI_ATTACH_CONTENT_LOCATION	= 0x3713;
const MAPI_ATTACH_FLAGS			= 0x3714;
const MAPI_DISPLAY_TYPE			= 0x3900;
const MAPI_TEMPLATEID			= 0x3902;
const MAPI_SMTP_ADDRESS			= 0x39FE;
const MAPI_ACCOUNT			= 0x3A00;
const MAPI_CONTACT_CALLBACK_PHONE	= 0x3A02;
const MAPI_GENERATION			= 0x3A05;
const MAPI_GIVEN_NAME			= 0x3A06;
const MAPI_CONTACT_BUSINESS_PHONE	= 0x3A08;
const MAPI_CONTACT_HOME_PHONE		= 0x3A09;
const MAPI_INITIALS			= 0x3A0A;
const MAPI_KEYWORD			= 0x3A0B;
const MAPI_LANGUAGE			= 0x3A0C;
const MAPI_LOCATION			= 0x3A0D;
const MAPI_SURNAME			= 0x3A11;
const MAPI_COMPANY_NAME			= 0x3A16;
const MAPI_TITLE			= 0x3A17;
const MAPI_DEPARTMENT_NAME		= 0x3A18;
const MAPI_CONTACT_PRIMARY_PHONE	= 0x3A1A;
const MAPI_CONTACT_BUSINESS_PHONE_2	= 0x3A1B;
const MAPI_CONTACT_OTHER_PHONE		= 0x3A1F;
const MAPI_CONTACT_MOBILE_PHONE		= 0x3A08;
const MAPI_CONTACT_BUSINESS_FAX		= 0x3A24;
const MAPI_COUNTRY			= 0x3A26;
const MAPI_LOCALITY			= 0x3A27;
const MAPI_STATE_PROVINCE		= 0x3A28;
const MAPI_HOME_PHONE_2			= 0x3A2F;
const MAPI_CONTACT_SEND_RICH_INFO	= 0x3A40;
const MAPI_CONTACT_MIDDLE_NAME		= 0x3A44;
const MAPI_DISPLAY_NAME_PREFIX		= 0x3A45;
const MAPI_CONTACT_COMPUTER_NETWORK	= 0x3A49;
const MAPI_CONTACT_TDD_PHONE		= 0x3A4B;
const MAPI_CONTACT_COMPANY_MAIN_PHONE	= 0x3A57;
const MAPI_CONTACT_CHILDRENS_NAMES	= 0x3A58;
const MAPI_INTERNET_CPID		= 0x3FDE;
const MAPI_CONFLICT_ENTRY_ID		= 0x3FF0;
const MAPI_MESSAGE_LOCALE_ID		= 0x3FF1;
const MAPI_CREATOR_NAME			= 0x3FF8;
const MAPI_CREATOR_ENTRY_ID		= 0x3FF9;
const MAPI_LAST_MODIFIER_NAME		= 0x3FFA;
const MAPI_LAST_MODIFIER_ENTRY_ID	= 0x3FFB;
const MAPI_CODE_PAGE			= 0x3FFD;
const MAPI_SENDER_FLAGS			= 0x4019;
const MAPI_SENT_REPRESENTING_FLAGS	= 0x401A;
const MAPI_SENDER_SIMPLE_DISPLAY_NAME	= 0x4030;
const MAPI_SENT_REPRESENTING_SIMPLE_DISPLAY_NAME = 0x4031;
const MAPI_CREATOR_SIMPLE_DISPLAY_NAME	= 0x4038;
const MAPI_LAST_MODIFER_SIMPLE_DISPLAY_NAME = 0x4039;
const MAPI_CONTENT_FILTER_SPAM_CONFIDENCE = 0x4076;
const MAPI_INET_MAIL_OVERRIDE_FORMAT	= 0x5902;
const MAPI_MSG_EDITOR_FORMAT		= 0x5909;
const MAPI_EXCEPTION_REPLACE_TIME	= 0x7FF9;
const MAPI_ATTACHMENT_LINK_ID		= 0x7FFA;
const MAPI_EXCEPTION_START_TIME		= 0x7FFB;
const MAPI_EXCEPTION_END_TIME		= 0x7FFC;
const MAPI_ATTACHMENT_FLAGS		= 0x7FFD;
const MAPI_ATTACHMENT_HIDDEN		= 0x7FFE;
const MAPI_ATTACHMENT_CONTACT_PHOTO	= 0x7FFF;
const MAPI_CONTACT_FILE_UNDER		= 0x8005;
const MAPI_CONTACT_LAST_AND_FIRST_NAME	= 0x8017;
const MAPI_CONTACT_COMPANY_AND_FULL_NAME = 0x8018;
const MAPI_ADDRESS_BOOK_PROVIDER_EMAIL_LIST = 0x8028;
const MAPI_ADDRESS_BOOK_PROVIDER_ARRAY_TYPE = 0x8029;
const MAPI_CONTACT_EMAIL_1_FULL		= 0x8080;
const MAPI_CONTACT_EMAIL_1_ADDRESS_TYPE	= 0x8082;
const MAPI_CONTACT_EMAIL_1_ADDRESS	= 0x8083;
const MAPI_CONTACT_EMAIL_1_DISPLAY_NAME = 0x8084;
const MAPI_CONTACT_EMAIL_1_ENTRY_ID	= 0x8085;
const MAPI_CONTACT_EMAIL_2_FULL		= 0x8090;
const MAPI_CONTACT_EMAIL_2_ADDRESS_TYPE	= 0x8092;
const MAPI_CONTACT_EMAIL_2_ADDRESS	= 0x8093;
const MAPI_CONTACT_EMAIL_2_DISPLAY_NAME = 0x8094;
const MAPI_CONTACT_EMAIL_2_ENTRY_ID	= 0x8095;
const MAPI_CONTACT_COUNTRY_CODE		= 0x80DD;
const MAPI_TASK_STATUS			= 0x8101;
const MAPI_TASK_PERCENT_COMPLETE	= 0x8102;
const MAPI_TASK_TEAM_TASK		= 0x8103;
const MAPI_TASK_START_DATE		= 0x8104;
const MAPI_TASK_DUE_DATE		= 0x8105;
const MAPI_TASK_RESET_REMINDER		= 0x8107;
const MAPI_TASK_ACCEPTED		= 0x8108;
const MAPI_TASK_DEAD_OCCURRENCE		= 0x8109;
const MAPI_TASK_DATE_COMPLETED		= 0x810F;
const MAPI_TASK_ACTUAL_EFFORT		= 0x8110;
const MAPI_TASK_ESTIMATED_EFFORT	= 0x8111;
const MAPI_TASK_VERSION			= 0x8112;
const MAPI_TASK_STATE			= 0x8113;
const MAPI_TASK_LAST_UPDATE		= 0x8115;
const MAPI_TASK_RECURRENCE		= 0x8116;
const MAPI_TASK_ASSIGNERS		= 0x8117;
const MAPI_TASK_STATUS_ON_COMPLETE	= 0x8119;
const MAPI_TASK_HISTORY			= 0x811A;
const MAPI_TASK_UPDATES			= 0x811B;
const MAPI_TASK_COMPLETE		= 0x811C;
const MAPI_TASK_F_CREATOR		= 0x811E;
const MAPI_TASK_MULTIPLE_RECIPIENTS	= 0X8120;
const MAPI_TASK_ASSIGNER		= 0x8121;
const MAPI_TASK_LAST_USER		= 0x8122;
const MAPI_TASK_ORDINAL			= 0x8123;
const MAPI_TASK_NO_COMPUTE		= 0x8124;
const MAPI_TASK_LAST_DELEGATE		= 0x8125;
const MAPI_TASK_F_RECURRING		= 0x8126;
const MAPI_TASK_ROLE			= 0x8127;
const MAPI_TASK_OWNERSHIP		= 0x8129;
const MAPI_TASK_ACCEPTANCE_STATE	= 0x812A;
const MAPI_TASK_F_FIX_OFFLINE		= 0x812C;
const MAPI_APPOINTMENT_SEND_AS_ICAL	= 0x8200;
const MAPI_APPOINTMENT_SEQUENCE		= 0x8201;
const MAPI_APPOINTMENT_SEQUENCE_TIME	= 0x8202;
const MAPI_APPOINTMENT_LAST_SEQUENCE	= 0x8203;
const MAPI_APPOINTMENT_CHANGE_HIGHLIGHT	= 0x8204;
const MAPI_APPOINTMENT_BUSY_STATUS	= 0x8205; // (0 = free, 1 = tentative, 2 = busy, 3 out of office)
const MAPI_APPOINTMENT_AUXILIARY_FLAGS	= 0x8207;
const MAPI_APPOINTMENT_LOCATION		= 0x8208;
const MAPI_APPOINTMENT_START_WHOLE	= 0x820D;
const MAPI_APPOINTMENT_END_WHOLE	= 0x820E;
const MAPI_APPOINTMENT_END_DATE		= 0x8211;
const MAPI_APPOINTMENT_START_DATE	= 0x8212;
const MAPI_APPOINTMENT_DURATION		= 0x8213;
const MAPI_APPOINTMENT_COLOR		= 0x8214; // (1 = red, 2 = blue, 3 = green, 4 = grey, 5 = orange, 6 = cyan, 7 = olive, 8 = purple, 9 = teal, a = yellow)
const MAPI_APPOINTMENT_ALL_DAY		= 0x8215; // subtype
const MAPI_APPOINTMENT_RECUR		= 0x8216;
const MAPI_APPOINTMENT_STATE_FLAGS	= 0x8217; // (1 = M = meeting, 2 = R = received, 3 = C = canceled)
const MAPI_APPOINTMENT_RESPONSE_STATUS	= 0x8218; // (0 = not req'd, 1 = organized, 2 = tentative, 3 = accepted, 4 = declined, 5 = not responded)
const MAPI_APPOINTMENT_REPLY_TIME	= 0x8230;
const MAPI_APPOINTMENT_RECURRENCE_SET	= 0x8223; // Recurring (0 not, 1 daily, 2 weekly, etc)
const MAPI_APPOINTMENT_INTENDED_BUSY_STATUS = 0x8224; // (0 = free, 1 = tentative, 2 = busy, 3 out of office)
const MAPI_APPOINTMENT_F_INVITED	= 0x8229;
const MAPI_APPOINTMENT_REPLY_NAME	= 0x8230;
const MAPI_APPOINTMENT_RECURRENCE_TYPE	= 0x8231;
const MAPI_APPOINTMENT_RECURRENCE_PATTERN = 0x8232;
const MAPI_APPOINTMENT_RECURRENCE_TIMEZONE = 0x8234;
const MAPI_APPOINTMENT_RECURRENCE_START = 0x8235;
const MAPI_APPOINTMENT_RECURRENCE_END	= 0x8236;
const MAPI_APPOINTMENT_ALL_ATTENDEES	= 0x8238;
const MAPI_APPOINTMENT_AUTO_FILL_LOCATION = 0x823A;
const MAPI_APPOINTMENT_REQUIRED_ATTENDEES = 0x823B;
const MAPI_APPOINTMENT_OPTIONAL_ATTENDEES = 0x823C;
const MAPI_APPOINTMENT_UNSENDABLE_RECIPIENTS = 0x823D;
const MAPI_APPOINTMENT_CONFERENCING_CHECK = 0x8240;
const MAPI_APPOINTMENT_CONFERENCING_TYPE = 0x8241;
const MAPI_APPOINTMENT_DIRECTORY	= 0x8242;
const MAPI_APPOINTMENT_ORGANIZER_ALIAS	= 0x8243;
const MAPI_APPOINTMENT_AUTO_START_CHECK	= 0x8244;
const MAPI_APPOINTMENT_AUTO_START_WHEN	= 0x8245;
const MAPI_APPOINTMENT_ALLOW_EXTERNAL_CHECK = 0x8246;
const MAPI_APPOINTMENT_COLLABORATE_DOC	= 0x8247;
const MAPI_APPOINTMENT_NET_SHOW_URL	= 0x8248;
const MAPI_APPOINTMENT_ONLINE_PASSWORD	= 0x8249;
const MAPI_APPOINTMENT_PROPOSED_START_WHOLE = 0x8250;
const MAPI_APPOINTMENT_PROPOSED_END_WHOLE = 0x8251;
const MAPI_APPOINTMENT_TZ_START_DISPLAY	= 0x825E;
const MAPI_APPOINTMENT_TZ_END_DISPLAY	= 0x825F;
const MAPI_APPOINTMENT_PROPOSED_DURATION = 0x8256;
const MAPI_APPOINTMENT_COUNTER_PROPOSAL	= 0x8257;
const MAPI_APPOINTMENT_PROPOSAL_NUMBER	= 0x8259;
const MAPI_APPOINTMENT_NOT_ALLOW_PROPOSE = 0x825A;
const MAPI_COMMON_REMINDER_MINUTES_BEFORE = 0x8501;
const MAPI_COMMON_REMINDER_DATE		= 0x8502;
const MAPI_COMMON_REMINDER_SET		= 0x8503;
const MAPI_COMMON_PRIVATE		= 0x8506;
const MAPI_COMMON_AGING_DONT_AGE	= 0x850E;
const MAPI_COMMON_ALLOW_EXCEPTION_DELETE = 0x8510;
const MAPI_COMMON_START			= 0x8516;
const MAPI_COMMON_END			= 0x8517;
const MAPI_TASK_MODE			= 0x8518;
const MAPI_COMMON_REMINDER_OVERRIDE	= 0x851C;
const MAPI_COMMON_REMINDER_TYPE		= 0x851D;
const MAPI_COMMON_REMINDER_PLAY_SOUND	= 0x851E;
const MAPI_COMMON_COMPANIES		= 0x8539;
const MAPI_COMMON_CONTACTS		= 0x853A;
const MAPI_NEXT_REMINDER_TIME		= 0x8560; // Reminder date
const MAPI_USE_TNEF			= 0x8582;
const MAPI_COMMON_CONTACT_LINK_NAME	= 0x8586;
const MAPI_COMMON_CLASSIFIED		= 0x85B5;
const MAPI_COMMON_CLASSIFICATION	= 0x85B6;
const MAPI_COMMON_CLASSIFICATION_DESCRIPTION = 0x85B7;
const MAPI_COMMON_CLASSIFICATION_GUID	= 0x85B8;
const MAPI_COMMON_CLASSIFICATION_KEEP	= 0x85BA;


// convertion regex
//   from: const MAPI_\([^ \t]*\)[ \t].*$
//   to:   case MAPI_\1: return "\1";


function mattr_name_to_string( mattr_name, msg_class ) {
  if( msg_class && (msg_class.substring( 0, 15 ) == "IPM.Appointment" ||
		    msg_class.substring( 0, 12 ) == "IPM.Schedule" ||
		    msg_class.substring( 0, 22 ) == "IPM.Microsoft Schedule") ) {
    switch( mattr_name ) {
    case MAPI_MEETING_ATTENDEE_CRITICAL_CHANGE: return "MEETING_ATTENDEE_CRITICAL_CHANGE";
    case MAPI_MEETING_WHERE: return "MEETING_WHERE";
    case MAPI_MEETING_GLOBAL_OBJECT_ID: return "MEETING_GLOBAL_OBJECT_ID";
    case MAPI_MEETING_IS_SILENT: return "MEETING_IS_SILENT";
    case MAPI_MEETING_IS_RECURRING: return "MEETING_IS_RECURRING";
    case MAPI_MEETING_REQUIRED_ATTENDEES: return "MEETING_REQUIRED_ATTENDEES";
    case MAPI_MEETING_OPTIONAL_ATTENDEES: return "MEETING_OPTIONAL_ATTENDEES";
    case MAPI_MEETING_RESOURCE_ATTENDEES: return "MEETING_RESOURCE_ATTENDEES";
    case MAPI_MEETING_DELEGATE_MAIL: return "MEETING_DELEGATE_MAIL";
    case MAPI_MEETING_IS_EXCEPTION: return "MEETING_IS_EXCEPTION";
    case MAPI_MEETING_SINGLE_INVITE: return "MEETING_SINGLE_INVITE";
    case MAPI_MEETING_TIME_ZONE: return "MEETING_TIME_ZONE";
    case MAPI_MEETING_MONTH_INTERVAL: return "MEETING_MONTH_INTERVAL";
    case MAPI_MEETING_OWNER_CRITICAL_CHANGE: return "MEETING_OWNER_CRITICAL_CHANGE";
    case MAPI_MEETING_CALENDAR_TYPE: return "MEETING_CALENDAR_TYPE";
    case MAPI_MEETING_ALL_ATTENDEES: return "MEETING_ALL_ATTENDEES";
    case MAPI_MEETING_MESSAGE_CLASS: return "MEETING_MESSAGE_CLASS";
    case MAPI_MEETING_TYPE: return "MEETING_TYPE";
    case MAPI_MEETING_OLD_LOCATION: return "MEETING_OLD_LOCATION";
    }
  }
  
  switch( mattr_name ) {
  case MAPI_ACKNOWLEDGEMENT_MODE: return "ACKNOWLEDGEMENT_MODE";
  case MAPI_TEMPLATE_DATA: return "TEMPLATE_DATA";
  case MAPI_MEETING_ATTENDEE_CRITICAL_CHANGE: return "MEETING_ATTENDEE_CRITICAL_CHANGE";
  case MAPI_ALTERNATE_RECEIPIANT_ALLOWED: return "ALTERNATE_RECEIPIANT_ALLOWED";
  case MAPI_MEETING_WHERE: return "MEETING_WHERE";
  case MAPI_AUTHORIZING_USERS: return "AUTHORIZING_USERS";
  case MAPI_MEETING_GLOBAL_OBJECT_ID: return "MEETING_GLOBAL_OBJECT_ID";
  case MAPI_AUTO_FORWARD_COMMENT: return "AUTO_FORWARD_COMMENT";
  case MAPI_SCRIPT_DATA: return "SCRIPT_DATA";
  case MAPI_MEETING_IS_SILENT: return "MEETING_IS_SILENT";
  case MAPI_AUTO_FORWARDED: return "AUTO_FORWARDED";
  case MAPI_MEETING_IS_RECURRING: return "MEETING_IS_RECURRING";
  case MAPI_CONTENT_CONFIDENTIALITY_ALGORITHM_ID: return "CONTENT_CONFIDENTIALITY_ALGORITHM_ID";
  case MAPI_MEETING_REQUIRED_ATTENDEES: return "MEETING_REQUIRED_ATTENDEES";
  case MAPI_CONTENT_CORRELATOR: return "CONTENT_CORRELATOR";
  case MAPI_MEETING_OPTIONAL_ATTENDEES: return "MEETING_OPTIONAL_ATTENDEES";
  case MAPI_CONTENT_IDENTIFIER: return "CONTENT_IDENTIFIER";
  case MAPI_MEETING_RESOURCE_ATTENDEES: return "MEETING_RESOURCE_ATTENDEES";
  case MAPI_MEETING_DELEGATE_MAIL: return "MEETING_DELEGATE_MAIL";
  case MAPI_CONTENT_RETURN_REQUESTED: return "CONTENT_RETURN_REQUESTED";
  case MAPI_MEETING_IS_EXCEPTION: return "MEETING_IS_EXCEPTION";
  case MAPI_CONVERSATION_KEY: return "CONVERSATION_KEY";
  case MAPI_MEETING_SINGLE_INVITE: return "MEETING_SINGLE_INVITE";
  case MAPI_MEETING_TIME_ZONE: return "MEETING_TIME_ZONE";
  case MAPI_CONVERSION_EITS: return "CONVERSION_EITS";
  case MAPI_DEFERRED_DELIVERY_TIME: return "DEFERRED_DELIVERY_TIME";
  case MAPI_DELIVER_TIME: return "DELIVER_TIME";
  case MAPI_DISCARD_REASON: return "DISCARD_REASON";
  case MAPI_DISCLOSURE_OF_RECIPIENTS: return "DISCLOSURE_OF_RECIPIENTS";
  case MAPI_DL_EXPANSION_HISTORY: return "DL_EXPANSION_HISTORY";
  case MAPI_MEETING_MONTH_INTERVAL: return "MEETING_MONTH_INTERVAL";
  case MAPI_DL_EXPANSION_PROHIBITED: return "DL_EXPANSION_PROHIBITED";
  case MAPI_EXPIRY_TIME: return "EXPIRY_TIME";
  case MAPI_IMPLICIT_CONVERSION_PROHIBITED: return "IMPLICIT_CONVERSION_PROHIBITED";
  case MAPI_IMPORTANCE: return "IMPORTANCE";
  case MAPI_IPM_ID: return "IPM_ID";
  case MAPI_LATEST_DELIVERY_TIME: return "LATEST_DELIVERY_TIME";
  case MAPI_MESSAGE_CLASS: return "MESSAGE_CLASS";
  case MAPI_MEETING_OWNER_CRITICAL_CHANGE: return "MEETING_OWNER_CRITICAL_CHANGE";
  case MAPI_MESSAGE_DELIVERY_ID: return "MESSAGE_DELIVERY_ID";
  case MAPI_MEETING_CALENDAR_TYPE: return "MEETING_CALENDAR_TYPE";
  case MAPI_MEETING_ALL_ATTENDEES: return "MEETING_ALL_ATTENDEES";
  case MAPI_MESSAGE_SECURITY_LABEL: return "MESSAGE_SECURITY_LABEL";
  case MAPI_OBSOLETED_IPMS: return "OBSOLETED_IPMS";
  case MAPI_ORIGINALLY_INTENDED_RECIPIENT_NAME: return "ORIGINALLY_INTENDED_RECIPIENT_NAME";
  case MAPI_ORIGINAL_EITS: return "ORIGINAL_EITS";
  case MAPI_ORIGINATOR_CERTIFICATE: return "ORIGINATOR_CERTIFICATE";
  case MAPI_ORIGINATOR_DELIVERY_REPORT_REQUESTED: return "ORIGINATOR_DELIVERY_REPORT_REQUESTED";
  case MAPI_MEETING_CLEAN_GLOBAL_OBJECT_ID: return "MEETING_CLEAN_GLOBAL_OBJECT_ID";
  case MAPI_ORIGINATOR_RETURN_ADDRESS: return "ORIGINATOR_RETURN_ADDRESS";
  case MAPI_MEETING_MESSAGE_CLASS: return "MEETING_MESSAGE_CLASS";
  case MAPI_PARENT_KEY: return "PARENT_KEY";
  case MAPI_PRIORITY: return "PRIORITY";
  case MAPI_MEETING_TYPE: return "MEETING_TYPE";
  case MAPI_MEETING_OLD_LOCATION: return "MEETING_OLD_LOCATION";
  case MAPI_READ_RECEIPT_REQUESTED: return "READ_RECEIPT_REQUESTED";
  case MAPI_RECIPIENT_REASSIGNMENT_PROHIBITED: return "RECIPIENT_REASSIGNMENT_PROHIBITED";
  case MAPI_RELATED_IPMS: return "RELATED_IPMS";
  case MAPI_ORIGINAL_SENSITIVITY: return "ORIGINAL_SENSITIVITY";
  case MAPI_REPORT_TAG: return "REPORT_TAG";
  case MAPI_SENSITIVITY: return "SENSITIVITY";
  case MAPI_SUBJECT: return "SUBJECT";
  case MAPI_CLIENT_SUBMIT_TIME: return "CLIENT_SUBMIT_TIME";
  case MAPI_SUBJECT_PREFIX: return "SUBJECT_PREFIX";
  case MAPI_RECEIVED_BY_NAME: return "RECEIVED_BY_NAME";
  case MAPI_SENT_REPRESENTING_ENTRYID: return "SENT_REPRESENTING_ENTRYID";
  case MAPI_SENT_REPRESENTING_NAME: return "SENT_REPRESENTING_NAME";
  case MAPI_RCVD_REPRESENTING_ENTRYID: return "RCVD_REPRESENTING_ENTRYID";
  case MAPI_RCVD_REPRESENTING_NAME: return "RCVD_REPRESENTING_NAME";
  case MAPI_REPORT_ENTRYID: return "REPORT_ENTRYID";
  case MAPI_READ_RECEIPT_ENTRYID: return "READ_RECEIPT_ENTRYID";
  case MAPI_MESSAGE_SUBMISSION_ID: return "MESSAGE_SUBMISSION_ID";
  case MAPI_PROVIDER_SUBMIT_TIME: return "PROVIDER_SUBMIT_TIME";
  case MAPI_ORIGINAL_SUBJECT: return "ORIGINAL_SUBJECT";
  case MAPI_DISC_VAL: return "DISC_VAL";
  case MAPI_ORIG_MESSAGE_CLASS: return "ORIG_MESSAGE_CLASS";
  case MAPI_ORIGINAL_AUTHOR_ENTRYID: return "ORIGINAL_AUTHOR_ENTRYID";
  case MAPI_ORIGINAL_AUTHOR_NAME: return "ORIGINAL_AUTHOR_NAME";
  case MAPI_MESSAGE_TO_ME: return "MESSAGE_TO_ME";
  case MAPI_MESSAGE_CC_ME: return "MESSAGE_CC_ME";
  case MAPI_START_DATE: return "START_DATE";
  case MAPI_END_DATE: return "END_DATE";
  case MAPI_OWNER_APPT_ID: return "OWNER_APPT_ID";
  case MAPI_RESPONSE_REQUESTED: return "RESPONSE_REQUESTED";
  case MAPI_SENT_REPRESENTING_ADDR_TYPE: return "SENT_REPRESENTING_ADDR_TYPE";
  case MAPI_SENT_REPRESENTING_EMAIL: return "SENT_REPRESENTING_EMAIL";
  case MAPI_CONVERSATION_TOPIC: return "CONVERSATION_TOPIC";
  case MAPI_CONVERSATION_INDEX: return "CONVERSATION_INDEX";
  case MAPI_RECEIVED_BY_ADDRTYPE: return "RECEIVED_BY_ADDRTYPE";
  case MAPI_RECEIVED_BY_EMAIL_ADDRESS: return "RECEIVED_BY_EMAIL_ADDRESS";
  case MAPI_TRANSPORT_MESSAGE_HEADERS: return "TRANSPORT_MESSAGE_HEADERS";
  case MAPI_TNEF_CORRELATION_KEY: return "TNEF_CORRELATION_KEY";
  case MAPI_EXPLICIT_CONVERSION: return "EXPLICIT_CONVERSION";
  case MAPI_MESSAGE_TOKEN: return "MESSAGE_TOKEN";
  case MAPI_REPLY_REQUESTED: return "REPLY_REQUESTED";
  case MAPI_SENDER_NAME: return "SENDER_NAME";
  case MAPI_SENDER_SEARCH_KEY: return "SENDER_SEARCH_KEY";
  case MAPI_SENDER_EMAIL_ADDRESS: return "SENDER_EMAIL_ADDRESS";
  case MAPI_DELETE_AFTER_SUBMIT: return "DELETE_AFTER_SUBMIT";
  case MAPI_DISPLAY_BCC: return "DISPLAY_BCC";
  case MAPI_DISPLAY_CC: return "DISPLAY_CC";
  case MAPI_DISPLAY_TO: return "DISPLAY_TO";
  case MAPI_PARENT_DISPLAY: return "PARENT_DISPLAY";
  case MAPI_MESSAGE_DELIVERY_TIME: return "MESSAGE_DELIVERY_TIME";
  case MAPI_MESSAGE_FLAGS: return "MESSAGE_FLAGS";
  case MAPI_MESSAGE_SIZE: return "MESSAGE_SIZE";
  case MAPI_PARENT_ENTRYID: return "PARENT_ENTRYID";
  case MAPI_SENTMAIL_ENTRYID: return "SENTMAIL_ENTRYID";
  case MAPI_MESSAGE_RECIPIENTS: return "MESSAGE_RECIPIENTS";
  case MAPI_MESSAGE_ATTACHMENTS: return "MESSAGE_ATTACHMENTS";
  case MAPI_SUBMIT_FLAGS: return "SUBMIT_FLAGS";
  case MAPI_HASATTACH: return "HASATTACH";
  case MAPI_BODY_CRC: return "BODY_CRC";
  case MAPI_NORMALIZED_SUBJECT: return "NORMALIZED_SUBJECT";
  case MAPI_RTF_IN_SYNC: return "RTF_IN_SYNC";
  case MAPI_ATTACH_SIZE: return "ATTACH_SIZE";
  case MAPI_ATTACH_NUM: return "ATTACH_NUM";
  case MAPI_PREPROCESS: return "PREPROCESS";
  case MAPI_PRIMARY_SEND_ACCOUNT: return "PRIMARY_SEND_ACCOUNT";
  case MAPI_NEXT_SEND_ACCOUNT: return "NEXT_SEND_ACCOUNT";
  case MAPI_ACCESS: return "ACCESS";
  case MAPI_ROW_TYPE: return "ROW_TYPE";
  case MAPI_INSTANCE_KEY: return "INSTANCE_KEY";
  case MAPI_ACCESS_LEVEL: return "ACCESS_LEVEL";
  case MAPI_MAPPING_SIGNATURE: return "MAPPING_SIGNATURE";
  case MAPI_RECORD_TYPE: return "RECORD_TYPE";
  case MAPI_MESSAGE_TYPE: return "MESSAGE_TYPE";
  case MAPI_OBJECT_TYPE: return "OBJECT_TYPE";
  case MAPI_ENTRY_ID: return "ENTRY_ID";
  case MAPI_BODY: return "BODY";
  case MAPI_REPORT_TEXT: return "REPORT_TEXT";
  case MAPI_RTF_SYNC_BODY_CRC: return "RTF_SYNC_BODY_CRC";
  case MAPI_RTF_SYNC_BODY_COUNT: return "RTF_SYNC_BODY_COUNT";
  case MAPI_RTF_SYNC_BODY_TAG: return "RTF_SYNC_BODY_TAG";
  case MAPI_RTF_COMPRESSED: return "RTF_COMPRESSED";
  case MAPI_RTF_SYNC_PREFIX_COUNT: return "RTF_SYNC_PREFIX_COUNT";
  case MAPI_RTF_SYNC_TRAILING_COUNT: return "RTF_SYNC_TRAILING_COUNT";
  case MAPI_BODY_HTML: return "BODY_HTML";
  case MAPI_SMTP_MESSAGE_ID: return "SMTP_MESSAGE_ID";
  case MAPI_CONFLICT_ITEMS: return "CONFLICT_ITEMS";
  case MAPI_URL_COMP_NAME: return "URL_COMP_NAME";
  case MAPI_ATTRIBUTE_READ_ONLY: return "ATTRIBUTE_READ_ONLY";
  case MAPI_CATEGORIES: return "CATEGORIES";
  case MAPI_DISPLAY_NAME: return "DISPLAY_NAME";
  case MAPI_EMAIL_ADDRESS: return "EMAIL_ADDRESS";
  case MAPI_CREATION_TIME: return "CREATION_TIME";
  case MAPI_MODIFICATION_TIME: return "MODIFICATION_TIME";
  case MAPI_SEARCH_KEY: return "SEARCH_KEY";
  case MAPI_STORE_SUPPORT_MASK: return "STORE_SUPPORT_MASK";
  case MAPI_MDB_PROVIDER: return "MDB_PROVIDER";
  case MAPI_CONTAINER_FLAGS: return "CONTAINER_FLAGS";
  case MAPI_FOLDER_TYPE: return "FOLDER_TYPE";
  case MAPI_CONTENT_COUNT: return "CONTENT_COUNT";
  case MAPI_CONTENT_UNREAD: return "CONTENT_UNREAD";
  case MAPI_CONTAINER_HIERARCHY: return "CONTAINER_HIERARCHY";
  case MAPI_CONTAINER_CONTENTS: return "CONTAINER_CONTENTS";
  case MAPI_FOLDER_ASSOCIATED_CONTENTS: return "FOLDER_ASSOCIATED_CONTENTS";
  case MAPI_CONTAINER_CLASS: return "CONTAINER_CLASS";
  case MAPI_CONTAINER_MODIFY_VERSION: return "CONTAINER_MODIFY_VERSION";
  case MAPI_ATTACHMENT_X400_PARAMETERS: return "ATTACHMENT_X400_PARAMETERS";
  case MAPI_ATTACH_DATA_OBJ: return "ATTACH_DATA_OBJ";
  case MAPI_ATTACH_ENCODING: return "ATTACH_ENCODING";
  case MAPI_ATTACH_EXTENSION: return "ATTACH_EXTENSION";
  case MAPI_ATTACH_FILENAME: return "ATTACH_FILENAME";
  case MAPI_ATTACH_METHOD: return "ATTACH_METHOD";
  case MAPI_ATTACH_LONG_FILENAME: return "ATTACH_LONG_FILENAME";
  case MAPI_ATTACH_PATHNAME: return "ATTACH_PATHNAME";
  case MAPI_ATTACH_RENDERING: return "ATTACH_RENDERING";
  case MAPI_ATTACH_TAG: return "ATTACH_TAG";
  case MAPI_ATTACHMENT_RENDERING_POSITION: return "ATTACHMENT_RENDERING_POSITION";
  case MAPI_ATTACH_TRANSPORT_NAME: return "ATTACH_TRANSPORT_NAME";
  case MAPI_ATTACH_LONG_PATHNAME: return "ATTACH_LONG_PATHNAME";
  case MAPI_ATTACH_MIME_TAG: return "ATTACH_MIME_TAG";
  case MAPI_ATTACH_ADDITIONAL_INFO: return "ATTACH_ADDITIONAL_INFO";
  case MAPI_ATTACH_MIME_SEQUENCE: return "ATTACH_MIME_SEQUENCE";
  case MAPI_ATTACH_CONTENT_ID: return "ATTACH_CONTENT_ID";
  case MAPI_ATTACH_CONTENT_LOCATION: return "ATTACH_CONTENT_LOCATION";
  case MAPI_ATTACH_FLAGS: return "ATTACH_FLAGS";
  case MAPI_DISPLAY_TYPE: return "DISPLAY_TYPE";
  case MAPI_TEMPLATEID: return "TEMPLATEID";
  case MAPI_ACCOUNT: return "ACCOUNT";
  case MAPI_CONTACT_CALLBACK_PHONE: return "CONTACT_CALLBACK_PHONE";
  case MAPI_GENERATION: return "GENERATION";
  case MAPI_GIVEN_NAME: return "GIVEN_NAME";
  case MAPI_CONTACT_BUSINESS_PHONE: return "CONTACT_BUSINESS_PHONE";
  case MAPI_CONTACT_HOME_PHONE: return "CONTACT_HOME_PHONE";
  case MAPI_INITIALS: return "INITIALS";
  case MAPI_KEYWORD: return "KEYWORD";
  case MAPI_LANGUAGE: return "LANGUAGE";
  case MAPI_LOCATION: return "LOCATION";
  case MAPI_SURNAME: return "SURNAME";
  case MAPI_COMPANY_NAME: return "COMPANY_NAME";
  case MAPI_TITLE: return "TITLE";
  case MAPI_DEPARTMENT_NAME: return "DEPARTMENT_NAME";
  case MAPI_CONTACT_PRIMARY_PHONE: return "CONTACT_PRIMARY_PHONE";
  case MAPI_CONTACT_BUSINESS_PHONE_2: return "CONTACT_BUSINESS_PHONE_2";
  case MAPI_CONTACT_OTHER_PHONE: return "CONTACT_OTHER_PHONE";
  case MAPI_CONTACT_MOBILE_PHONE: return "CONTACT_MOBILE_PHONE";
  case MAPI_CONTACT_BUSINESS_FAX: return "CONTACT_BUSINESS_FAX";
  case MAPI_COUNTRY: return "COUNTRY";
  case MAPI_LOCALITY: return "LOCALITY";
  case MAPI_STATE_PROVINCE: return "STATE_PROVINCE";
  case MAPI_HOME_PHONE_2: return "HOME_PHONE_2";
  case MAPI_CONTACT_SEND_RICH_INFO: return "CONTACT_SEND_RICH_INFO";
  case MAPI_CONTACT_MIDDLE_NAME: return "CONTACT_MIDDLE_NAME";
  case MAPI_DISPLAY_NAME_PREFIX: return "DISPLAY_NAME_PREFIX";
  case MAPI_CONTACT_COMPUTER_NETWORK: return "CONTACT_COMPUTER_NETWORK";
  case MAPI_CONTACT_TDD_PHONE: return "CONTACT_TDD_PHONE";
  case MAPI_CONTACT_COMPANY_MAIN_PHONE: return "CONTACT_COMPANY_MAIN_PHONE";
  case MAPI_CONTACT_CHILDRENS_NAMES: return "CONTACT_CHILDRENS_NAMES";
  case MAPI_INTERNET_CPID: return "INTERNET_CPID";
  case MAPI_CONFLICT_ENTRY_ID: return "CONFLICT_ENTRY_ID";
  case MAPI_MESSAGE_LOCALE_ID: return "MESSAGE_LOCALE_ID";
  case MAPI_CREATOR_NAME: return "CREATOR_NAME";
  case MAPI_CREATOR_ENTRY_ID: return "CREATOR_ENTRY_ID";
  case MAPI_LAST_MODIFIER_NAME: return "LAST_MODIFIER_NAME";
  case MAPI_LAST_MODIFIER_ENTRY_ID: return "LAST_MODIFIER_ENTRY_ID";
  case MAPI_CODE_PAGE: return "CODE_PAGE";
  case MAPI_SENDER_FLAGS: return "SENDER_FLAGS";
  case MAPI_SENT_REPRESENTING_FLAGS: return "SENT_REPRESENTING_FLAGS";
  case MAPI_SENDER_SIMPLE_DISPLAY_NAME: return "SENDER_SIMPLE_DISPLAY_NAME";
  case MAPI_SENT_REPRESENTING_SIMPLE_DISPLAY_NAME: return "SENT_REPRESENTING_SIMPLE_DISPLAY_NAME";
  case MAPI_CREATOR_SIMPLE_DISPLAY_NAME: return "CREATOR_SIMPLE_DISPLAY_NAME";
  case MAPI_LAST_MODIFER_SIMPLE_DISPLAY_NAME: return "LAST_MODIFER_SIMPLE_DISPLAY_NAME";
  case MAPI_CONTENT_FILTER_SPAM_CONFIDENCE: return "CONTENT_FILTER_SPAM_CONFIDENCE";
  case MAPI_INET_MAIL_OVERRIDE_FORMAT: return "INET_MAIL_OVERRIDE_FORMAT";
  case MAPI_MSG_EDITOR_FORMAT: return "MSG_EDITOR_FORMAT";
  case MAPI_EXCEPTION_REPLACE_TIME: return "EXCEPTION_REPLACE_TIME";
  case MAPI_ATTACHMENT_LINK_ID: return "ATTACHMENT_LINK_ID";
  case MAPI_EXCEPTION_START_TIME: return "EXCEPTION_START_TIME";
  case MAPI_EXCEPTION_END_TIME: return "EXCEPTION_END_TIME";
  case MAPI_ATTACHMENT_FLAGS: return "ATTACHMENT_FLAGS";
  case MAPI_ATTACHMENT_HIDDEN: return "ATTACHMENT_HIDDEN";
  case MAPI_ATTACHMENT_CONTACT_PHOTO: return "ATTACHMENT_CONTACT_PHOTO";
  case MAPI_CONTACT_FILE_UNDER: return "CONTACT_FILE_UNDER";
  case MAPI_CONTACT_LAST_AND_FIRST_NAME: return "CONTACT_LAST_AND_FIRST_NAME";
  case MAPI_CONTACT_COMPANY_AND_FULL_NAME: return "CONTACT_COMPANY_AND_FULL_NAME";
  case MAPI_ADDRESS_BOOK_PROVIDER_EMAIL_LIST: return "ADDRESS_BOOK_PROVIDER_EMAIL_LIST";
  case MAPI_ADDRESS_BOOK_PROVIDER_ARRAY_TYPE: return "ADDRESS_BOOK_PROVIDER_ARRAY_TYPE";
  case MAPI_CONTACT_EMAIL_1_FULL: return "CONTACT_EMAIL_1_FULL";
  case MAPI_CONTACT_EMAIL_1_ADDRESS_TYPE: return "CONTACT_EMAIL_1_ADDRESS_TYPE";
  case MAPI_CONTACT_EMAIL_1_ADDRESS: return "CONTACT_EMAIL_1_ADDRESS";
  case MAPI_CONTACT_EMAIL_1_DISPLAY_NAME: return "CONTACT_EMAIL_1_DISPLAY_NAME";
  case MAPI_CONTACT_EMAIL_1_ENTRY_ID: return "CONTACT_EMAIL_1_ENTRY_ID";
  case MAPI_CONTACT_EMAIL_2_FULL: return "CONTACT_EMAIL_2_FULL";
  case MAPI_CONTACT_EMAIL_2_ADDRESS_TYPE: return "CONTACT_EMAIL_2_ADDRESS_TYPE";
  case MAPI_CONTACT_EMAIL_2_ADDRESS: return "CONTACT_EMAIL_2_ADDRESS";
  case MAPI_CONTACT_EMAIL_2_DISPLAY_NAME: return "CONTACT_EMAIL_2_DISPLAY_NAME";
  case MAPI_CONTACT_EMAIL_2_ENTRY_ID: return "CONTACT_EMAIL_2_ENTRY_ID";
  case MAPI_CONTACT_COUNTRY_CODE: return "CONTACT_COUNTRY_CODE";
  case MAPI_TASK_STATUS: return "TASK_STATUS";
  case MAPI_TASK_PERCENT_COMPLETE: return "TASK_PERCENT_COMPLETE";
  case MAPI_TASK_TEAM_TASK: return "TASK_TEAM_TASK";
  case MAPI_TASK_START_DATE: return "TASK_START_DATE";
  case MAPI_TASK_DUE_DATE: return "TASK_DUE_DATE";
  case MAPI_TASK_RESET_REMINDER: return "TASK_RESET_REMINDER";
  case MAPI_TASK_ACCEPTED: return "TASK_ACCEPTED";
  case MAPI_TASK_DEAD_OCCURRENCE: return "TASK_DEAD_OCCURRENCE";
  case MAPI_TASK_DATE_COMPLETED: return "TASK_DATE_COMPLETED";
  case MAPI_TASK_ACTUAL_EFFORT: return "TASK_ACTUAL_EFFORT";
  case MAPI_TASK_ESTIMATED_EFFORT: return "TASK_ESTIMATED_EFFORT";
  case MAPI_TASK_VERSION: return "TASK_VERSION";
  case MAPI_TASK_STATE: return "TASK_STATE";
  case MAPI_TASK_LAST_UPDATE: return "TASK_LAST_UPDATE";
  case MAPI_TASK_RECURRENCE: return "TASK_RECURRENCE";
  case MAPI_TASK_ASSIGNERS: return "TASK_ASSIGNERS";
  case MAPI_TASK_STATUS_ON_COMPLETE: return "TASK_STATUS_ON_COMPLETE";
  case MAPI_TASK_HISTORY: return "TASK_HISTORY";
  case MAPI_TASK_UPDATES: return "TASK_UPDATES";
  case MAPI_TASK_COMPLETE: return "TASK_COMPLETE";
  case MAPI_TASK_F_CREATOR: return "TASK_F_CREATOR";
  case MAPI_TASK_MULTIPLE_RECIPIENTS: return "TASK_MULTIPLE_RECIPIENTS";
  case MAPI_TASK_ASSIGNER: return "TASK_ASSIGNER";
  case MAPI_TASK_LAST_USER: return "TASK_LAST_USER";
  case MAPI_TASK_ORDINAL: return "TASK_ORDINAL";
  case MAPI_TASK_NO_COMPUTE: return "TASK_NO_COMPUTE";
  case MAPI_TASK_LAST_DELEGATE: return "TASK_LAST_DELEGATE";
  case MAPI_TASK_F_RECURRING: return "TASK_F_RECURRING";
  case MAPI_TASK_ROLE: return "TASK_ROLE";
  case MAPI_TASK_OWNERSHIP: return "TASK_OWNERSHIP";
  case MAPI_TASK_ACCEPTANCE_STATE: return "TASK_ACCEPTANCE_STATE";
  case MAPI_TASK_F_FIX_OFFLINE: return "TASK_F_FIX_OFFLINE";
  case MAPI_APPOINTMENT_SEND_AS_ICAL: return "APPOINTMENT_SEND_AS_ICAL";
  case MAPI_APPOINTMENT_SEQUENCE: return "APPOINTMENT_SEQUENCE";
  case MAPI_APPOINTMENT_SEQUENCE_TIME: return "APPOINTMENT_SEQUENCE_TIME";
  case MAPI_APPOINTMENT_LAST_SEQUENCE: return "APPOINTMENT_LAST_SEQUENCE";
  case MAPI_APPOINTMENT_CHANGE_HIGHLIGHT: return "APPOINTMENT_CHANGE_HIGHLIGHT";
  case MAPI_APPOINTMENT_BUSY_STATUS: return "APPOINTMENT_BUSY_STATUS";
  case MAPI_APPOINTMENT_AUXILIARY_FLAGS: return "APPOINTMENT_AUXILIARY_FLAGS";
  case MAPI_APPOINTMENT_LOCATION: return "APPOINTMENT_LOCATION";
  case MAPI_APPOINTMENT_START_WHOLE: return "APPOINTMENT_START_WHOLE";
  case MAPI_APPOINTMENT_END_WHOLE: return "APPOINTMENT_END_WHOLE";
  case MAPI_APPOINTMENT_END_DATE: return "APPOINTMENT_END_DATE";
  case MAPI_APPOINTMENT_START_DATE: return "APPOINTMENT_START_DATE";
  case MAPI_APPOINTMENT_DURATION: return "APPOINTMENT_DURATION";
  case MAPI_APPOINTMENT_COLOR: return "APPOINTMENT_COLOR";
  case MAPI_APPOINTMENT_ALL_DAY: return "APPOINTMENT_ALL_DAY";
  case MAPI_APPOINTMENT_RECUR: return "APPOINTMENT_RECUR";
  case MAPI_APPOINTMENT_STATE_FLAGS: return "APPOINTMENT_STATE_FLAGS";
  case MAPI_APPOINTMENT_RESPONSE_STATUS: return "APPOINTMENT_RESPONSE_STATUS";
  case MAPI_APPOINTMENT_REPLY_TIME: return "APPOINTMENT_REPLY_TIME";
  case MAPI_APPOINTMENT_RECURRENCE_SET: return "APPOINTMENT_RECURRENCE_SET";
  case MAPI_APPOINTMENT_INTENDED_BUSY_STATUS: return "APPOINTMENT_INTENDED_BUSY_STATUS";
  case MAPI_APPOINTMENT_F_INVITED: return "APPOINTMENT_F_INVITED";
  case MAPI_APPOINTMENT_REPLY_NAME: return "APPOINTMENT_REPLY_NAME";
  case MAPI_APPOINTMENT_RECURRENCE_TYPE: return "APPOINTMENT_RECURRENCE_TYPE";
  case MAPI_APPOINTMENT_RECURRENCE_PATTERN: return "APPOINTMENT_RECURRENCE_PATTERN";
  case MAPI_APPOINTMENT_RECURRENCE_TIMEZONE: return "APPOINTMENT_RECURRENCE_TIMEZONE";
  case MAPI_APPOINTMENT_RECURRENCE_START: return "APPOINTMENT_RECURRENCE_START";
  case MAPI_APPOINTMENT_RECURRENCE_END: return "APPOINTMENT_RECURRENCE_END";
  case MAPI_APPOINTMENT_ALL_ATTENDEES: return "APPOINTMENT_ALL_ATTENDEES";
  case MAPI_APPOINTMENT_AUTO_FILL_LOCATION: return "APPOINTMENT_AUTO_FILL_LOCATION";
  case MAPI_APPOINTMENT_REQUIRED_ATTENDEES: return "APPOINTMENT_REQUIRED_ATTENDEES";
  case MAPI_APPOINTMENT_OPTIONAL_ATTENDEES: return "APPOINTMENT_OPTIONAL_ATTENDEES";
  case MAPI_APPOINTMENT_UNSENDABLE_RECIPIENTS: return "APPOINTMENT_UNSENDABLE_RECIPIENTS";
  case MAPI_APPOINTMENT_CONFERENCING_CHECK: return "APPOINTMENT_CONFERENCING_CHECK";
  case MAPI_APPOINTMENT_CONFERENCING_TYPE: return "APPOINTMENT_CONFERENCING_TYPE";
  case MAPI_APPOINTMENT_DIRECTORY: return "APPOINTMENT_DIRECTORY";
  case MAPI_APPOINTMENT_ORGANIZER_ALIAS: return "APPOINTMENT_ORGANIZER_ALIAS";
  case MAPI_APPOINTMENT_AUTO_START_CHECK: return "APPOINTMENT_AUTO_START_CHECK";
  case MAPI_APPOINTMENT_AUTO_START_WHEN: return "APPOINTMENT_AUTO_START_WHEN";
  case MAPI_APPOINTMENT_ALLOW_EXTERNAL_CHECK: return "APPOINTMENT_ALLOW_EXTERNAL_CHECK";
  case MAPI_APPOINTMENT_COLLABORATE_DOC: return "APPOINTMENT_COLLABORATE_DOC";
  case MAPI_APPOINTMENT_NET_SHOW_URL: return "APPOINTMENT_NET_SHOW_URL";
  case MAPI_APPOINTMENT_ONLINE_PASSWORD: return "APPOINTMENT_ONLINE_PASSWORD";
  case MAPI_APPOINTMENT_PROPOSED_START_WHOLE: return "APPOINTMENT_PROPOSED_START_WHOLE";
  case MAPI_APPOINTMENT_PROPOSED_END_WHOLE: return "APPOINTMENT_PROPOSED_END_WHOLE";
  case MAPI_APPOINTMENT_TZ_START_DISPLAY: return "APPOINTMENT_TZ_START_DISPLAY";
  case MAPI_APPOINTMENT_TZ_END_DISPLAY: return "APPOINTMENT_TZ_END_DISPLAY";
  case MAPI_APPOINTMENT_PROPOSED_DURATION: return "APPOINTMENT_PROPOSED_DURATION";
  case MAPI_APPOINTMENT_COUNTER_PROPOSAL: return "APPOINTMENT_COUNTER_PROPOSAL";
  case MAPI_APPOINTMENT_PROPOSAL_NUMBER: return "APPOINTMENT_PROPOSAL_NUMBER";
  case MAPI_APPOINTMENT_NOT_ALLOW_PROPOSE: return "APPOINTMENT_NOT_ALLOW_PROPOSE";
  case MAPI_COMMON_REMINDER_MINUTES_BEFORE: return "COMMON_REMINDER_MINUTES_BEFORE";
  case MAPI_COMMON_REMINDER_DATE: return "COMMON_REMINDER_DATE";
  case MAPI_COMMON_REMINDER_SET: return "COMMON_REMINDER_SET";
  case MAPI_COMMON_PRIVATE: return "COMMON_PRIVATE";
  case MAPI_COMMON_AGING_DONT_AGE: return "COMMON_AGING_DONT_AGE";
  case MAPI_COMMON_ALLOW_EXCEPTION_DELETE: return "COMMON_ALLOW_EXCEPTION_DELETE";
  case MAPI_COMMON_START: return "COMMON_START";
  case MAPI_COMMON_END: return "COMMON_END";
  case MAPI_TASK_MODE: return "TASK_MODE";
  case MAPI_COMMON_REMINDER_OVERRIDE: return "COMMON_REMINDER_OVERRIDE";
  case MAPI_COMMON_REMINDER_TYPE: return "COMMON_REMINDER_TYPE";
  case MAPI_COMMON_REMINDER_PLAY_SOUND: return "COMMON_REMINDER_PLAY_SOUND";
  case MAPI_COMMON_COMPANIES: return "COMMON_COMPANIES";
  case MAPI_COMMON_CONTACTS: return "COMMON_CONTACTS";
  case MAPI_NEXT_REMINDER_TIME: return "NEXT_REMINDER_TIME";
  case MAPI_USE_TNEF: return "USE_TNEF";
  case MAPI_COMMON_CONTACT_LINK_NAME: return "COMMON_CONTACT_LINK_NAME";
  case MAPI_COMMON_CLASSIFIED: return "COMMON_CLASSIFIED";
  case MAPI_COMMON_CLASSIFICATION: return "COMMON_CLASSIFICATION";
  case MAPI_COMMON_CLASSIFICATION_DESCRIPTION: return "COMMON_CLASSIFICATION_DESCRIPTION";
  case MAPI_COMMON_CLASSIFICATION_GUID: return "COMMON_CLASSIFICATION_GUID";
  case MAPI_COMMON_CLASSIFICATION_KEEP: return "COMMON_CLASSIFICATION_KEEP";

  default: return( to_hex( mattr_name, 2 ) );
  }
}

